import os
import sys
import requests
import logging
from dotenv import load_dotenv
import mysql.connector
from mysql.connector import Error
import re
import xml.etree.ElementTree as ET

# Add parent directory to path for imports
sys.path.append(os.path.dirname(os.path.dirname(os.path.abspath(__file__))))

# Load environment variables from .env in /apps
load_dotenv(os.path.join(os.path.dirname(os.path.dirname(os.path.abspath(__file__))), '.env'))

# Configure logging
logging.basicConfig(level=logging.INFO, format='%(asctime)s - %(levelname)s - %(message)s')
logger = logging.getLogger(__name__)

SENATE_URL = 'https://ilga.gov/ftp/Committees/104SenateCommitteeMembers.xml'
HOUSE_URL = 'https://ilga.gov/ftp/Committees/104HouseCommitteeMembers.xml'

# --- DB Connection ---
def get_db_connection():
    try:
        return mysql.connector.connect(
            host=os.getenv('DB_HOST'),
            database=os.getenv('DB_NAME'),
            user=os.getenv('DB_USER'),
            password=os.getenv('DB_PASS')
        )
    except Error as e:
        logger.error(f"Error connecting to database: {str(e)}")
        raise

# --- Table Creation ---
def setup_database():
    try:
        with get_db_connection() as connection:
            with connection.cursor() as cursor:
                cursor.execute('''
                    CREATE TABLE IF NOT EXISTS committees (
                        id INT AUTO_INCREMENT PRIMARY KEY,
                        name VARCHAR(255),
                        chamber VARCHAR(10),
                        code VARCHAR(20),
                        UNIQUE KEY unique_committee (name, chamber, code)
                    )
                ''')
                cursor.execute('''
                    CREATE TABLE IF NOT EXISTS committee_members (
                        id INT AUTO_INCREMENT PRIMARY KEY,
                        committee_id INT,
                        legislator_id INT,
                        role VARCHAR(50),
                        FOREIGN KEY (committee_id) REFERENCES committees(id),
                        FOREIGN KEY (legislator_id) REFERENCES legislators(id)
                    )
                ''')
                connection.commit()
                logger.info("Database tables for committees created/updated successfully")
    except Exception as e:
        logger.error(f"Error setting up database: {str(e)}")
        raise

# --- Legislator Lookup ---
def get_legislators():
    try:
        with get_db_connection() as connection:
            with connection.cursor(dictionary=True) as cursor:
                cursor.execute("SELECT id, name FROM legislators")
                return cursor.fetchall()
    except Exception as e:
        logger.error(f"Error fetching legislators: {str(e)}")
        return []

def normalize_name(name):
    # Remove punctuation, lower, strip, collapse spaces
    name = re.sub(r'[.,]', '', name)
    name = re.sub(r'\s+', ' ', name)
    return name.strip().lower()

def match_legislator_id(name, legislators):
    # Try to match by normalized name
    n = normalize_name(name)
    for leg in legislators:
        if normalize_name(leg['name']) == n:
            return leg['id']
    # Try last, first to first last
    if ',' in name:
        last, first = name.split(',', 1)
        first = first.strip().split(' ')[0]
        last = last.strip()
        n2 = normalize_name(f"{first} {last}")
        for leg in legislators:
            if normalize_name(leg['name']) == n2:
                return leg['id']
    return None

# --- Committee Parsing (XML) ---
def parse_committee_file_xml(text, chamber):
    committees = []
    try:
        root = ET.fromstring(text)
        for committee_elem in root.findall('Committee'):
            name = committee_elem.findtext('Name')
            code = committee_elem.findtext('Code')
            members = []
            for member_elem in committee_elem.findall('Member'):
                member_name = member_elem.findtext('Name')
                party = member_elem.findtext('Party')
                role = member_elem.findtext('Title') or 'Member'
                members.append({'name': member_name, 'party': party, 'role': role})
            committees.append({
                'name': name,
                'code': code,
                'chamber': chamber,
                'members': members
            })
    except Exception as e:
        logger.error(f"Error parsing XML for {chamber}: {e}")
    return committees

# --- Main Loader ---
def load_committees():
    setup_database()
    legislators = get_legislators()
    # Download files with error checking
    senate_resp = requests.get(SENATE_URL)
    senate_resp.raise_for_status()
    house_resp = requests.get(HOUSE_URL)
    house_resp.raise_for_status()
    # Try decoding with utf-8-sig to strip BOM if present
    try:
        senate_text = senate_resp.content.decode('utf-8-sig')
    except Exception:
        senate_text = senate_resp.text
    try:
        house_text = house_resp.content.decode('utf-8-sig')
    except Exception:
        house_text = house_resp.text
    logger.info(f"First 200 chars of Senate file: {repr(senate_text[:200])}")
    logger.info(f"First 200 chars of House file: {repr(house_text[:200])}")
    senate_committees = parse_committee_file_xml(senate_text, 'senate')
    house_committees = parse_committee_file_xml(house_text, 'house')
    logger.info(f"Parsed {len(senate_committees)} senate committees, {len(house_committees)} house committees")
    if senate_committees:
        logger.info(f"First senate committee: {senate_committees[0]}")
    else:
        logger.info("No senate committees parsed.")
    if house_committees:
        logger.info(f"First house committee: {house_committees[0]}")
    else:
        logger.info("No house committees parsed.")
    all_committees = senate_committees + house_committees
    with get_db_connection() as connection:
        with connection.cursor(dictionary=True) as cursor:
            for committee in all_committees:
                # Insert or get committee
                cursor.execute("""
                    SELECT id FROM committees WHERE name=%s AND chamber=%s AND code=%s
                """, (committee['name'], committee['chamber'], committee['code']))
                row = cursor.fetchone()
                if row:
                    committee_id = row['id']
                else:
                    cursor.execute("""
                        INSERT INTO committees (name, chamber, code) VALUES (%s, %s, %s)
                    """, (committee['name'], committee['chamber'], committee['code']))
                    committee_id = cursor.lastrowid
                # Remove old members for this committee
                cursor.execute("DELETE FROM committee_members WHERE committee_id=%s", (committee_id,))
                # Insert members
                for member in committee['members']:
                    legislator_id = match_legislator_id(member['name'], legislators)
                    if legislator_id:
                        cursor.execute("""
                            INSERT INTO committee_members (committee_id, legislator_id, role) VALUES (%s, %s, %s)
                        """, (committee_id, legislator_id, member['role']))
                    else:
                        logger.warning(f"No match for legislator: {member['name']} in committee {committee['name']}")
            connection.commit()
    logger.info("Committees and members loaded successfully.")

def main():
    load_committees()

if __name__ == "__main__":
    main() 