import requests
from bs4 import BeautifulSoup
import logging
import re
import json
import os
from typing import Dict, Tuple, List
from datetime import datetime
import sys

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

class SessionAnalyzer:
    def __init__(self):
        self.base_url = "https://ilga.gov/legislation/default.asp"
        
        # Get the absolute path to the project root directory
        self.project_root = os.path.dirname(os.path.dirname(os.path.dirname(os.path.abspath(__file__))))
        
        # Get GENERAL_ASSEMBLY from PHP config
        self.general_assembly = self.get_general_assembly()
        
        # Session IDs
        self.session_types = {
            "regular": 1,
            "special": 2
        }
        
        # Template for session data structure
        self.session_template = {
            "Senate": {
                "Bills": 0,
                "Resolutions": 0,
                "Joint Resolutions": 0,
                "Joint Resolution Constitutional Amendments": 0
            },
            "House": {
                "Bills": 0,
                "Resolutions": 0,
                "Joint Resolutions": 0,
                "Joint Resolution Constitutional Amendments": 0
            },
            "Other": {
                "Executive Orders": 0,
                "Joint Session Resolutions": 0,
                "Appointment Messages": 0
            }
        }
        
        self.session_data = {
            "regular": self._create_session_data(),
            "special": self._create_session_data(),
            "metadata": {
                "general_assembly": self.general_assembly,
                "last_updated": datetime.now().isoformat(),
                "base_url": self.base_url
            }
        }

    def _create_session_data(self):
        """Create a deep copy of the session template"""
        return json.loads(json.dumps(self.session_template))

    def get_config_content(self):
        """Read the PHP config file content"""
        config_path = os.path.join(self.project_root, 'app', 'config.php')
        try:
            with open(config_path, 'r') as f:
                return f.read()
        except FileNotFoundError:
            logger.error(f"Config file not found at: {config_path}")
            raise

    def get_general_assembly(self):
        """Read GENERAL_ASSEMBLY from PHP config file"""
        content = self.get_config_content()
        match = re.search(r'const GENERAL_ASSEMBLY = (\d+);', content)
        if match:
            return int(match.group(1))
        raise ValueError("Could not find GENERAL_ASSEMBLY in config.php")

    def get_max_number(self, doc_type, session_id):
        """Get the maximum number for a specific document type by checking bill listing pages"""
        try:
            # First get the session page
            url = f"{self.base_url}?GA={self.general_assembly}&SessionID={session_id}"
            response = requests.get(url)
            response.raise_for_status()
            soup = BeautifulSoup(response.text, 'lxml')
            
            # Find all links that could contain our document type
            links = soup.find_all('a', href=True)
            max_num = 0
            
            # Look for links containing grplist.asp and our document type
            for link in links:
                href = link['href']
                if 'grplist.asp' in href and f'DocTypeID={doc_type}' in href:
                    # Extract num1 and num2 from the URL
                    match = re.search(r'num1=(\d+)&num2=(\d+)', href)
                    if match:
                        end_num = int(match.group(2))
                        max_num = max(max_num, end_num)
            
            session_type = "regular" if session_id == 1 else "special"
            logger.info(f"Found maximum number for {doc_type} in {session_type} session: {max_num}")
            return max_num

        except Exception as e:
            logger.error(f"Error getting max number for {doc_type} in session {session_id}: {str(e)}")
            return 0

    def analyze_session(self):
        """Analyze both regular and special sessions and collect document counts"""
        try:
            # Process both regular and special sessions
            for session_type, session_id in self.session_types.items():
                logger.info(f"Analyzing {session_type} session...")
                session_data = self.session_data[session_type]
                
                # Senate Documents
                session_data["Senate"]["Bills"] = self.get_max_number("SB", session_id)
                session_data["Senate"]["Resolutions"] = self.get_max_number("SR", session_id)
                session_data["Senate"]["Joint Resolutions"] = self.get_max_number("SJR", session_id)
                session_data["Senate"]["Joint Resolution Constitutional Amendments"] = self.get_max_number("SJRCA", session_id)

                # House Documents
                session_data["House"]["Bills"] = self.get_max_number("HB", session_id)
                session_data["House"]["Resolutions"] = self.get_max_number("HR", session_id)
                session_data["House"]["Joint Resolutions"] = self.get_max_number("HJR", session_id)
                session_data["House"]["Joint Resolution Constitutional Amendments"] = self.get_max_number("HJRCA", session_id)

                # Other Documents
                session_data["Other"]["Executive Orders"] = self.get_max_number("EO", session_id)
                session_data["Other"]["Joint Session Resolutions"] = self.get_max_number("JSR", session_id)
                session_data["Other"]["Appointment Messages"] = self.get_max_number("AM", session_id)

            return self.session_data

        except Exception as e:
            logger.error(f"Error analyzing sessions: {str(e)}")
            raise

    def save_to_json(self, filename='session_config.json'):
        """Save the session data to a JSON file"""
        try:
            with open(filename, 'w') as f:
                json.dump(self.session_data, f, indent=2)
            logging.info(f"Session configuration saved to {filename}")
        except Exception as e:
            logging.error(f"Error saving session config: {str(e)}")
            raise

if __name__ == "__main__":
    try:
        analyzer = SessionAnalyzer()
        data = analyzer.analyze_session()
        analyzer.save_to_json()
        
        # Print summary
        print("\nSession Analysis Summary:")
        
        for session_type in ["regular", "special"]:
            print(f"\n{session_type.title()} Session:")
            session_data = data[session_type]
            
            print("\nSenate:")
            for key, value in session_data["Senate"].items():
                print(f" - {key}: {value}")
            
            print("\nHouse:")
            for key, value in session_data["House"].items():
                print(f" - {key}: {value}")
            
            print("\nOther:")
            for key, value in session_data["Other"].items():
                print(f" - {key}: {value}")

        print("\nSession analysis completed successfully. Starting bill listing collection...")
        
        # Import and run the bill listing collector
        try:
            import os
            import sys
            # Add the parent directory to sys.path so we can import from bills directory
            current_dir = os.path.dirname(os.path.abspath(__file__))
            parent_dir = os.path.dirname(current_dir)
            if parent_dir not in sys.path:
                sys.path.append(parent_dir)
            
            from bills.bill_listing_collector import BillListingCollector
            collector = BillListingCollector()
            collector.collect_all_listings()
            print("Bill listing collection completed successfully!")
        except Exception as e:
            logger.error(f"Error running bill listing collector: {e}")
            print("\nWarning: Session analysis completed, but bill listing collection failed.")
            print(f"Error: {str(e)}")
            sys.exit(1)
            
    except Exception as e:
        logger.error(f"Error in main: {e}")
        sys.exit(1) 