import os
import sys
import re
import logging
import mysql.connector
from mysql.connector import Error
import requests
from bs4 import BeautifulSoup
from datetime import datetime
from dotenv import load_dotenv

logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(levelname)s - %(message)s'
)

class BillListingExpander:
    def __init__(self):
        self.project_root = os.path.dirname(os.path.dirname(os.path.dirname(os.path.abspath(__file__))))
        self.setup_logging()
        self.db_config = self.get_database_config()
        
    def setup_logging(self):
        self.logger = logging.getLogger(__name__)
        self.logger.setLevel(logging.INFO)
        
    def get_database_config(self):
        """Read database configuration from .env file"""
        env_path = os.path.join(self.project_root, 'app', '.env')
        load_dotenv(env_path)
        try:
            return {
                'host': os.getenv('DB_HOST'),
                'user': os.getenv('DB_USER'),
                'password': os.getenv('DB_PASS'),
                'database': os.getenv('DB_NAME')
            }
        except Exception as e:
            self.logger.error(f"Failed to read database config from .env: {str(e)}")
            sys.exit(1)

    def connect_to_database(self):
        """Establish database connection"""
        try:
            conn = mysql.connector.connect(**self.db_config)
            return conn
        except Error as e:
            self.logger.error(f"Error connecting to database: {str(e)}")
            sys.exit(1)

    def create_expanded_table(self, cursor):
        """Create the expanded bill listings table if it doesn't exist"""
        create_table_sql = """
        CREATE TABLE IF NOT EXISTS bill_listings_exp (
            id INT AUTO_INCREMENT PRIMARY KEY,
            chamber VARCHAR(10) NOT NULL,
            link_name VARCHAR(255) NOT NULL,
            link_url VARCHAR(512) NOT NULL,
            doc_type VARCHAR(50) NOT NULL,
            session_type VARCHAR(50) NOT NULL,
            full_name VARCHAR(255) NOT NULL,
            bill_number VARCHAR(20) NOT NULL,
            bill_url VARCHAR(512) NOT NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            UNIQUE KEY unique_bill (bill_number, session_type)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
        """
        try:
            cursor.execute(create_table_sql)
            self.logger.info("Expanded table created or already exists")
        except Error as e:
            self.logger.error(f"Error creating expanded table: {str(e)}")
            sys.exit(1)

    def get_bill_listings(self, cursor):
        """Retrieve all bill listings from the original table"""
        try:
            cursor.execute("SELECT * FROM bill_listings")
            return cursor.fetchall()
        except Error as e:
            self.logger.error(f"Error retrieving bill listings: {str(e)}")
            return []

    def parse_bill_page(self, url):
        """Parse a bill listing page and extract individual bill information"""
        try:
            response = requests.get(url)
            response.raise_for_status()
            soup = BeautifulSoup(response.text, 'lxml')
            
            bills = []
            # Find all bill links in the page
            for link in soup.find_all('a', href=True):
                href = link.get('href', '')
                text = link.get_text(strip=True)
                
                # Look for bill links matching patterns like "HB1234" or "SB5678"
                bill_match = re.match(r'(HB|SB|HR|SR|HJR|SJR)(\d+)', text)
                if bill_match and 'BillStatus' in href:
                    bill_type, number = bill_match.groups()
                    
                    bills.append({
                        'full_name': text,
                        'bill_number': f"{bill_type}{number}",
                        'url': href if href.startswith('http') else f"https://ilga.gov{href}"
                    })
            
            return bills
        except Exception as e:
            self.logger.error(f"Error parsing bill page {url}: {str(e)}")
            return []

    def insert_expanded_listing(self, cursor, listing, bill_info):
        """Insert or update an expanded bill listing"""
        sql = """
        INSERT INTO bill_listings_exp 
        (chamber, link_name, link_url, doc_type, session_type, full_name, bill_number, bill_url)
        VALUES (%s, %s, %s, %s, %s, %s, %s, %s)
        ON DUPLICATE KEY UPDATE
        link_url = VALUES(link_url),
        bill_url = VALUES(bill_url),
        updated_at = CURRENT_TIMESTAMP
        """
        try:
            cursor.execute(sql, (
                listing['chamber'],
                listing['link_name'],
                listing['link_url'],
                listing['doc_type'],
                listing['session_type'],
                bill_info['full_name'],
                bill_info['bill_number'],
                bill_info['url']  # bill_url
            ))
            return True
        except Error as e:
            self.logger.error(f"Error inserting expanded listing: {str(e)}")
            return False

    def expand_listings(self):
        """Process all bill listings and expand them into individual entries"""
        conn = self.connect_to_database()
        cursor = conn.cursor(dictionary=True)
        
        try:
            self.create_expanded_table(cursor)
            listings = self.get_bill_listings(cursor)
            
            for listing in listings:
                self.logger.info(f"Processing listing: {listing['link_name']}")
                bills = self.parse_bill_page(listing['link_url'])
                
                for bill in bills:
                    if self.insert_expanded_listing(cursor, listing, bill):
                        self.logger.info(f"Successfully processed bill {bill['bill_number']}")
                
            conn.commit()
            
        except Exception as e:
            self.logger.error(f"Error in expand_listings: {str(e)}")
            conn.rollback()
        finally:
            cursor.close()
            conn.close()

    def run(self):
        """Main execution method"""
        self.logger.info("Starting bill listing expansion")
        self.expand_listings()
        self.logger.info("Completed bill listing expansion")

if __name__ == "__main__":
    expander = BillListingExpander()
    expander.run() 